/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
 */
package gov.nasa.worldwind.servers.wms.generators;

import gov.nasa.worldwind.formats.rpf.*;
import gov.nasa.worldwind.geom.Sector;
import gov.nasa.worldwind.servers.wms.*;
import gov.nasa.worldwind.servers.wms.formats.BufferedImageFormatter;
import gov.nasa.worldwind.servers.wms.formats.ImageFormatter;
import gov.nasa.worldwind.servers.wms.utilities.WaveletCodec;
import gov.nasa.worldwind.servers.wms.xml.*;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.lang.Exception;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

/**
 * A concrete implementation of MapGenerator that serves RPF dataseries (e.g., CADRG, CIB, etc.).
 * Any framefiles residing below the configured root-directory for this MapGenerator (see
 * {@link gov.nasa.worldwind.servers.wms.MapSource}) are identified. The framefiles that are
 * eligible to served can be constrained to specific RPF data-series classification via an
 * optional configuration property (see below); all framefiles are otherwise considered available.
 *
 * <p>The implementation also attempts to use wavelet encodings of these files to reconstruct
 * small-scale representations of the files. These encodings reside in files named after the
 * individual framefiles with a ".wvt" suffix appended. The encodings are presumed to be co-located with
 * the framefiles, unless otherwise specified with an optional configuration property (see below).
 * If the encodings are not co-located, they must reside in a directory structure that
 * parallels that of this MapGenerator's data root-directory.</p>
 *
 * <p>Several optional properties may be included in the XML configuration of the corresponding
 * {@link gov.nasa.worldwind.servers.wms.MapSource} element:
 *
 * <pre>
 *   &lt;!-- if a tile's footprint in a map request is below this size (in pixels),
 *        the image is reconstructed from a wavelet encoding --&gt;
 *   &lt;property name="wavelet_image_threshold" value="..." /&gt;
 *
 *   &lt;!-- amount of wavelet encodings to preload ( size in pixels, sq.), --&gt;
 *   &lt;property name="wavelet_preload_size" value="..." /&gt;
 *
 *   &lt;!-- root directory where the wavelet encodings reside; the encodings are
 *        otherwise presumed to be co-located with the image tiles. --&gt;
 *   &lt;property name="wavelet_encoding_root_dir" value="..." /&gt;
 *
 *   &lt;!-- constrain the framefiles to be served to this RPF dataseries --&gt;
 *   &lt;property name="data_series" value="..." /&gt;
 * </pre>
 *
 * @author brownrigg
 * @version $Id$
 */

public class RPFGenerator implements MapGenerator {


    public ServiceInstance getServiceInstance() { return new RPFServiceInstance(); }

    public boolean initialize(MapSource mapSource) throws IOException, WMSServiceException {
        boolean success = true;  // Assume the best...
        try {
            this.mapSource = mapSource;

            // Get these optional properties...
            Properties props = mapSource.getProperties();
            String srcName = mapSource.getName();

            this.smallImageSize = parseIntProperty(props, WAVELET_IMAGE_THRESHOLD, smallImageSize, srcName);
            int tmp = parseIntProperty(props, WAVELET_PRELOAD_SIZE, preloadRes, srcName);
            if (!WaveletCodec.isPowerOfTwo(tmp)) {
                SysLog.inst().warning(srcName + ": value given for \"" + WAVELET_PRELOAD_SIZE + "\" must be power of two");
                SysLog.inst().warning("  given: " + tmp + ",  overriding with default of: " + this.preloadRes);
            }
            else
                this.preloadRes = tmp;

            this.dataSeries = props.getProperty(RPF_DATA_SERIES);
            this.encodingRootDir = props.getProperty(WAVELET_ROOT_DIR);

            // Track down all the RPF frameFiles we can find...
            this.rootDir = mapSource.getRootDir();
            RPFCrawler crawler = new RPFCrawler();
            DataSeriesGrouper grouper = new DataSeriesGrouper(this.dataSeries);
            crawler.invoke(new File(this.rootDir), grouper, false);

            File[] frameFiles = grouper.getFrameFiles();
            if (!(frameFiles.length > 0)) {
                SysLog.inst().error("RPFGenerator: found no FrameFile's rooted under: " + this.rootDir);
                success = false;
            }

            // convert the list of Files into a more convenient form...
            processFrameFiles(frameFiles);

            // consolidate the individual bounds to get a global bound...
            consolidateBounds();

            // Report configuration out to log...
            dumpToLog(false);

            // Preload (perhaps partially) any framefiles we find. NOTE: we want to perform this *after*
            // any optional properties have been parsed, as some parameters are configurable.
            preloadWaveletFiles();

        }
        catch (Exception ex) {
            success = false;
            SysLog.inst().stackTrace(ex);
            throw new WMSServiceException(ex.toString() + ":" + ex.getMessage());
        }

        return success;
    }

    public Sector getBBox() {
        return globalBnds;
    }
    
    public String[] getCRS() {
        return new String[] {crsStr};
    }
 
    //
    // Preprocess the framefile "Files", as indentified by the RPFCrawler, into a more
    // suitable form.
    //
    private void processFrameFiles(File[] files) {
        this.frameFiles = new FrameFile[files.length];
        int rootLen = this.rootDir.length();

        for (int i=0; i<files.length; i++) {
            // remove the common path prefix (i.e., the "this.rootDir" portion...
            String parent = files[i].getParent();
            if (parent.regionMatches(0, this.rootDir, 0, rootLen))
                parent = parent.substring(rootLen);

            String file = files[i].getName();

            WMSRPFFrameFilename f = new WMSRPFFrameFilename(parent, file);
            this.frameFiles[i] = new FrameFile(f);
        }

    }

    //
    // Find the global bounds for this collection of frame files (i.e., the union of their Sectors).
    //
    private void consolidateBounds() {
        this.globalBnds = new Sector(this.frameFiles[0].file.getSector());
        for (int i=1; i<this.frameFiles.length; i++) {
            try {
                this.globalBnds = this.globalBnds.union(frameFiles[i].file.getSector());
            } catch (Exception e) {
                // We've observed that occasionally framefiles will exist that have "invalid"
                // names for a particular series & zone; don't want these to short circuit everything...
                ;
            }
        }
    }

    //
    // Preload any wavelet files associated with a framefile, to the desired resolution.
    //
    private void preloadWaveletFiles() {
        for (FrameFile frame : this.frameFiles) {
            try {
                WaveletCodec codec = WaveletCodec.loadPartially(getWaveletEncodingFile(frame), this.preloadRes);
                frame.codec = codec;
            } catch (IOException ex) { /* no guarantees the wavelet files exist */ }
        }
    }

    //
    // A little method to log info about the RPF collection(s) we've loaded.
    //
    private void dumpToLog(boolean dumpFrameFiles) {
        SysLog.inst().info("RootDir: " + this.rootDir);
        SysLog.inst().info("  bounds: " + this.globalBnds.toString());
        SysLog.inst().info("  wavlet preload size: " + this.preloadRes + ", wavelet-image generation threshold: "
            + this.smallImageSize);
        SysLog.inst().info("  num. frame files: " + this.frameFiles.length);
        
        if (dumpFrameFiles) {
            for (int i=0; i<this.frameFiles.length; i++) {
                SysLog.inst().info("     " + frameFiles[i].file.getPathToFilename());
            }
        }
    }

    //
    // Convenience method for parsing/validating optional configuration parameters.
    //
    private int parseIntProperty(Properties props, String key, int defaultVal, String name) {
        int retval = defaultVal;
        String tmp = props.getProperty(key);
        if (tmp != null) {
            try {
               retval = Integer.parseInt(tmp);
            }  catch (NumberFormatException ex) {
                SysLog.inst().warning("Could not decode '" + key +
                    "' property in config. for " + name);
            }
        }

        return retval;
    }

    //
    // Convenience method use by this class and its internal ServiceInstance for
    // computing pathname to wavelet-encoding location.
    //
    private File getWaveletEncodingFile(FrameFile frame) {
        String path = (this.encodingRootDir != null) ?
                this.encodingRootDir :
                this.rootDir;
        return new File(path + File.separator + frame.file.getPathToFilename() + WaveletCodec.WVT_EXT);
    }

    // --------------------------------------------
    // class ServiceInstance
    //
    // Used to manage per-request state.
    //
    public class RPFServiceInstance implements ServiceInstance {

        public ImageFormatter serviceRequest(WMSGetMapRequest req) throws IOException, WMSServiceException {
            try {
                // Identify which TOCs and frame files we'll need to satisfy this request...
                Sector reqSector = Sector.fromDegrees(req.getBBoxYMin(), req.getBBoxYMax(),
                        req.getBBoxXMin(), req.getBBoxXMax());

                BufferedImage reqImage = new BufferedImage(req.getWidth(), req.getHeight(), BufferedImage.TYPE_4BYTE_ABGR);
                Graphics2D g2d = (Graphics2D) reqImage.getGraphics();

                int numFramesInRequest = 0;
                int debugMinFrameRes = Integer.MAX_VALUE;
                int debugMaxFrameRes = -Integer.MAX_VALUE;

                for (FrameFile frame : RPFGenerator.this.frameFiles) {
                    try {
                        // The call to getSector() can throw an exception if the file is
                        // named with an inappropriate frameNumber for the dataseries/zone.
                        // Also, serving the 5MGlobalNavChart data has been known to throw NPEs
                        // because some of those files have parse errors.
                        // We don't want these to short circuit the entire request, so
                        // trap any such occurances and ignore 'em.
                        if (!reqSector.intersects(frame.file.getSector()))
                            continue;
                    } catch (Exception ex) {
                        /* ignore this framefile */
                        continue;
                    }

                    // Frame overlaps request; attempt to draw it...
                    ++numFramesInRequest;

                    Sector frameSector = frame.file.getSector();
                    Sector overlap = reqSector.intersection(frameSector);

                    // find size of the frame's footprint at the requested image resolution...
                    int footprintX = (int) (frameSector.getDeltaLonDegrees() * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
                    int footprintY = (int) (frameSector.getDeltaLatDegrees() * reqImage.getHeight() / reqSector.getDeltaLatDegrees());

                    // Depending upon footprint, either get image from it RPF framefile, or reconstruct
                    // it from a wavelet encoding.
                    BufferedImage sourceImage;
                    if (footprintX > smallImageSize || footprintY > smallImageSize) {
                        sourceImage = getImageFromRPFSource(frame);
                    } else {
                        int maxRes = footprintX;
                        maxRes = (footprintY > maxRes) ? footprintY : maxRes;
                        int power = (int) Math.ceil(Math.log(maxRes) / Math.log(2.));
                        int res = (int) Math.pow(2., power);
                        res = Math.max(1, res);

                        if (res < debugMinFrameRes) debugMinFrameRes = res;
                        if (res > debugMaxFrameRes) debugMaxFrameRes = res;

                        sourceImage = getImageFromWaveletEncoding(frame, res);
                    }

                    if (sourceImage == null) // failed to get an image...ignore this FrameFile
                        continue;

//                    // Destination subimage...
//                    int dx1 = (int) ((overlap.getMinLongitude().degrees - reqSector.getMinLongitude().degrees)
//                            * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
//                    int dx2 = (int) ((overlap.getMaxLongitude().degrees - reqSector.getMinLongitude().degrees)
//                            * reqImage.getWidth() / reqSector.getDeltaLonDegrees());
//                    int dy1 = (int) ((reqSector.getMaxLatitude().degrees - overlap.getMaxLatitude().degrees)
//                            * reqImage.getHeight() / reqSector.getDeltaLatDegrees());
//                    int dy2 = (int) ((reqSector.getMaxLatitude().degrees - overlap.getMinLatitude().degrees)
//                            * reqImage.getHeight() / reqSector.getDeltaLatDegrees());
//
//                    // Source subimage...
//                    int sx1 = (int) ((overlap.getMinLongitude().degrees - frameSector.getMinLongitude().degrees)
//                            * sourceImage.getWidth() / frameSector.getDeltaLonDegrees());
//                    int sx2 = (int) ((overlap.getMaxLongitude().degrees - frameSector.getMinLongitude().degrees)
//                            * sourceImage.getWidth() / frameSector.getDeltaLonDegrees());
//                    sx1 = Math.max(0, sx1);
//                    sx2 = Math.min(sourceImage.getWidth() - 1, sx2);
//
//                    int sy1 = (int) ((frameSector.getMaxLatitude().degrees - overlap.getMaxLatitude().degrees)
//                            * sourceImage.getHeight() / frameSector.getDeltaLatDegrees());
//                    int sy2 = (int) ((frameSector.getMaxLatitude().degrees - overlap.getMinLatitude().degrees)
//                            * sourceImage.getHeight() / frameSector.getDeltaLatDegrees());
//                    sy1 = Math.max(0, sy1);
//                    sy2 = Math.min(sourceImage.getHeight() - 1, sy2);
//
//                    g2d.drawImage(sourceImage, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, null);

                    double tx = (frameSector.getMinLongitude().degrees - reqSector.getMinLongitude().degrees) * (reqImage.getWidth() / reqSector.getDeltaLonDegrees());
                    double ty = (reqSector.getMaxLatitude().degrees - frameSector.getMaxLatitude().degrees) * (reqImage.getHeight() / reqSector.getDeltaLatDegrees());
                    double sx = (reqImage.getWidth() / reqSector.getDeltaLonDegrees()) * (frameSector.getDeltaLonDegrees() / sourceImage.getWidth());
                    double sy = (reqImage.getHeight() / reqSector.getDeltaLatDegrees()) * (frameSector.getDeltaLatDegrees() / sourceImage.getHeight());

                    AffineTransform xform = g2d.getTransform();
                    g2d.translate(tx, ty);
                    g2d.scale(sx, sy);
                    g2d.drawRenderedImage(sourceImage, null);
                    g2d.setTransform(xform);
                }

                SysLog.inst().info("   " + numFramesInRequest + " frames in req," +
                        " min/max footprint: " + debugMinFrameRes + ", " + debugMaxFrameRes);

                return new BufferedImageFormatter(reqImage);

            } catch (Exception ex) {
                SysLog.inst().stackTrace(ex);
                throw new WMSServiceException("RPF request failed: " + ex.getMessage());
            }
        }

        public List<File> serviceRequest(WMSGetImageryListRequest req) throws IOException, WMSServiceException {
            try {
                // Identify which frame files we'll need to satisfy this request...
                Sector reqSector = Sector.fromDegrees(req.getBBoxYMin(), req.getBBoxYMax(),
                        req.getBBoxXMin(), req.getBBoxXMax());

                List<File> files = new ArrayList<File>(100);
                int numFramesInRequest = 0;

                for (FrameFile frame : RPFGenerator.this.frameFiles) {
                    try {
                        // The call to getSector() can throw an exception if the file is
                        // named with an inappropriate frameNumber for the dataseries/zone.
                        // Also, serving the 5MGlobalNavChart data has been known to throw NPEs
                        // because some of those files have parse errors.
                        // We don't want these to short circuit the entire request, so
                        // trap any such occurances and ignore 'em.
                        if (!reqSector.intersects(frame.file.getSector()))
                            continue;
                    } catch (Exception ex) {
                        /* ignore this framefile */
                        continue;
                    }

                    // Frame overlaps request; add it to our list...
                    ++numFramesInRequest;
                    File f = new File(RPFGenerator.this.rootDir + File.separator + frame.file.getPathToFilename());
                    files.add(f);
                }

                return files;

            } catch (Exception ex) {
                SysLog.inst().stackTrace(ex);
                throw new WMSServiceException("RPF GetImageryListRequest failed: " + ex.getMessage());
            }
        }



        //
        // Attempts to return the specified FrameFile as a BufferedImage. Returns null on failure.
        //
        private BufferedImage getImageFromRPFSource(FrameFile frame) {
            BufferedImage sourceImage = null;
            try {
                RPFImageFile sourceFile = RPFImageFile.load(new File(RPFGenerator.this.rootDir + File.separator + frame.file.getPathToFilename()));
                sourceImage = sourceFile.getBufferedImage();
            } catch (Exception ex) {
                SysLog.inst().warning("Failed to load frame file " + frame.file.getPathToFilename() + ": " + ex.toString());
            }

            return sourceImage;
        }
        
        //
        // Attempts to reconstruct the given FrameFile as a BufferedImage from a WaveletEncoding.
        // Returns null if encoding does not exist or on any other failure.
        //
        private BufferedImage getImageFromWaveletEncoding(FrameFile frame, int resolution) {
            BufferedImage sourceImage = null;
            try {

                if (resolution <= 0 || frame.codec == null)
                    return sourceImage;

                WaveletCodec codec = null;
                if (resolution <= RPFGenerator.this.preloadRes)
                    codec = frame.codec;
                else
                    // read wavelet file...
                    codec = WaveletCodec.loadPartially(getWaveletEncodingFile(frame), resolution);


                if (codec != null)
                    sourceImage = codec.reconstruct(resolution);

            } catch (Exception ex) {
                SysLog.inst().stackTrace(ex);
                SysLog.inst().warning("Failed to reconstruct wavelet from " + frame.file.getPathToFilename() + ": " + ex.toString());
            }

            return sourceImage;
        }

        public void freeResources() { /* No-op */ }
    }

    // ----------------------------------------------------
    // class DataSeriesGrouper
    //
    // RPFCrawler.Grouper that identifies framefiles of a given dataseries.
    //
    private class DataSeriesGrouper extends RPFCrawler.RPFGrouper {

        public DataSeriesGrouper(String dataSeries) {
            super(RPFFrameProperty.DATA_SERIES);
            this.dataSeries = ("".equals(dataSeries)) ? null : dataSeries;
        }
        public void addToGroup(Object groupKey, File rpfFile, RPFFrameFilename rpfFrameFilename) {
            if (this.dataSeries == null || this.dataSeries.equals((String) groupKey))
                workList.add(rpfFile);
        }
        public File[] getFrameFiles() {
            File[] files = new File[workList.size()];
            return workList.toArray(files);
        }
        
        private String dataSeries;
        private List<File> workList = new ArrayList<File>(500);
    }

    // ----------------------------------------------------
    // class WMSRPFFrameFilename
    //
    // This class wraps a RPFFrameFilename, adding a reference to the frame file's
    // "transform" and its bounding rectangle (both of which are resolved lazily).
    //
    private static class WMSRPFFrameFilename {

        public WMSRPFFrameFilename(String path, String filename) throws IllegalArgumentException {
            this.filename = RPFFrameFilename.parseFilename(filename.toUpperCase());
            this.path = path + File.separator + filename;
        }

        public Sector getSector() {
            if (this.bounds == null) {
                RPFFrameTransform trans = getFrameTransform();
                this.bounds = trans.computeFrameCoverage(this.filename.getFrameNumber());
            }
            return this.bounds;
        }

        public RPFFrameTransform getFrameTransform() {
            if (this.transform == null) {
                RPFDataSeries dataSeries = RPFDataSeries.dataSeriesFor(this.filename.getDataSeriesCode());
                this.transform = RPFFrameTransform.createFrameTransform(this.filename.getZoneCode(),
                        dataSeries.rpfDataType, dataSeries.scaleOrGSD);
            }
            return this.transform;
        }

        public String getPathToFilename() {
            return path;
        }

        private RPFFrameFilename filename = null;
        private String path = null;
        private Sector bounds = null;
        private RPFFrameTransform transform = null;
    }

    // -----------------------------------------------
    // class FrameFile
    //
    // A small private class to bundle info about framefiles.
    // Public access to fields in intentional.
    //
    private static class FrameFile {
        public WMSRPFFrameFilename file;
        public WaveletCodec        codec;

        public FrameFile(WMSRPFFrameFilename file, WaveletCodec codec) {
            this.file = file;
            this.codec = codec;
        }

        public FrameFile(WMSRPFFrameFilename file) {
            this.file = file;
            this.codec = null;
        }
    }

    private MapSource mapSource = null;
    private String rootDir;
    private String encodingRootDir = null;
    private FrameFile frameFiles[];
    private Sector globalBnds;
    private String dataSeries = null;

    // performance tuning parameters...
    private int smallImageSize = 256;
    private int preloadRes = 32;

    // Configuration property keys...
    private static final String crsStr="EPSG:4326";
    private static final String WAVELET_IMAGE_THRESHOLD = "wavelet_image_threshold";
    private static final String WAVELET_PRELOAD_SIZE = "wavelet_preload_size";
    private static final String WAVELET_ROOT_DIR = "wavelet_encoding_root_dir";
    private static final String RPF_DATA_SERIES = "data_series";
}
